/*
Google Charts JSF Wrapper
Copyright (C) 2016  Alex Rhodes
https://www.alexscottrhodes.com

Information about this project including set-up and configuration information can be found here:
https://bitbucket.org/alexscottrhodes/googlechartjsf

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0
	
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

 2016 "Google" and the Google logo are registered trademarks of Google Inc.
The Google Charts API is used under the Apache 2.0 License above. Google software 
is loaded from Google's servers as required in the Google Terms of Service.
*/
package com.alexscottrhodes.chartModel;


import java.util.ArrayList;
import java.util.HashMap;

import com.alexscottrhodes.constructionModel.ChartSeries;
import com.alexscottrhodes.enums.AxisType;
import com.alexscottrhodes.enums.CurveType;
import com.alexscottrhodes.enums.LegendPosition;
/**
 * Model of the Google Combo chart object
 * @author Alex Rhodes
 *
 */
public class GoogleComboChart {
	/**
	 * ID that will be provided by the JSF build cycle
	 */
	private String id;
	/**
	 * The title of the chart
	 */
	private String title;
	/**
	 * The width of the chart in pixels, provided in-line in the  JSF tag with the "width" parameter, excluding "px". The width can also be provided
	 * in the backing bean, and will supersede the in-line width.
	 */
	private int width;
	/**
	 * The height of the chart in pixels, provided in-line in the  JSF tag with the "height" parameter, excluding "px". The height can also be provided
	 * in the backing bean, and will supersede the in-line height.
	 */
	private int height;
	/**
	 * A y-value to display as a straight line across the chart to indicate some index. 
	 */
	private int indexLineVal;
	/**
	 * A label for the index line
	 */
	private String indexLineLabel;
	/**
	 * A list of series to add to the chart, individual series should be added with the 
		{@link #addSeries(ChartSeries) addSeries} method. 
	 */
	private ArrayList<ChartSeries> seriesList;
	/**
	 * A list of axes to add to the chart
	 */
	private HashMap<AxisType, ArrayList<ChartSeries>> axes;
	/**
	 * A color value to use for the chart's title. The value is a string that can represent any valid HTML or CSS color
	 */
	private String titleColor;
	/**
	 * Boolean to indicate a bold title
	 */
	private boolean titleBold;
	/**
	 * A label for the vertical axis
	 */
	private String verticalAxisLabel;
	/**
	 * The style of chart line, as defined in the Google Chart API.
	 *  See <a href="https://developers.google.com/chart/glossary">Google Charts API</a>
	 */
	private CurveType curveType = CurveType.DEFAULT;
	
	/**
	 * The LegendPosition to place the legend when rendering the chart. 
	 * See <a href="https://developers.google.com/chart/glossary">Google Charts API</a>
	 */
	private LegendPosition legendPosition = LegendPosition.RIGHT;
	
	/**
	 * A boolean for Bar type charts to indicate if the bars should be stacekd or adjacent.
	 */
	private boolean stacked;
	
	public GoogleComboChart(){
		seriesList = new ArrayList<ChartSeries>();
		axes = new HashMap<AxisType, ArrayList<ChartSeries>>();
		axes.put(AxisType.X, new ArrayList<ChartSeries>());
		axes.put(AxisType.Y, new ArrayList<ChartSeries>());
	}
	/**
	 * Create an index line to be displayed across the chart
	 * @param value an Int of the Y value to draw the index line
	 * @param label a String to display as the label for the index line
	 */
	public void setIndexLine(int value, String label) {
		this.indexLineVal = value;
		this.indexLineLabel = label;
	}
	
	/**
	 * Add a data series to the chart
	 * @param series a ChartSeries to be added to the chart
	 */
	public void addSeries(ChartSeries series){
		seriesList.add(series);
	}
	/**
	 * Add a data series to be used as an axis
	 * @param series a ChartSeries object to be added as an axis
	 * @param axis an AxisType specifying which axis to use the series on
	 */
	public void addAxisSeries(ChartSeries series, AxisType axis){
		axes.get(axis).add(series);
	}
	
	//Getters and Setters 
	 
	public String getId() {
		return id;
	}
	public void setId(String id) {
		this.id = id;
	}
	public String getTitle() {
		return title;
	}
	public void setTitle(String title) {
		this.title = title;
	}
	public int getWidth() {
		return width;
	}
	public void setWidth(int width) {
		this.width = width;
	}
	public int getHeight() {
		return height;
	}
	public void setHeight(int height) {
		this.height = height;
	}
	public ArrayList<ChartSeries> getSeriesList() {
		return seriesList;
	}
	public void setSeriesList(ArrayList<ChartSeries> seriesList) {
		this.seriesList = seriesList;
	}

	public HashMap<AxisType, ArrayList<ChartSeries>> getAxes() {
		return axes;
	}

	public void setAxes(HashMap<AxisType, ArrayList<ChartSeries>> axes) {
		this.axes = axes;
	}

	public boolean isTitleBold() {
		return titleBold;
	}

	public void setTitleBold(boolean titleBold) {
		this.titleBold = titleBold;
	}

	public String getTitleColor() {
		return titleColor;
	}

	public void setTitleColor(String titleColor) {
		this.titleColor = titleColor;
	}

	public String getVerticalAxisLabel() {
		return verticalAxisLabel;
	}

	public void setVerticalAxisLabel(String verticalAxisLabel) {
		this.verticalAxisLabel = verticalAxisLabel;
	}
	
	public String getIndexLineLabel() {
		return indexLineLabel;
	}

	public int getIndexLineVal() {
		return indexLineVal;
	}
	public CurveType getCurveType() {
		return curveType;
	}
	public void setCurveType(CurveType curveType) {
		this.curveType = curveType;
	}
	public LegendPosition getLegendPosition() {
		return legendPosition;
	}
	public void setLegendPosition(LegendPosition legendPosition) {
		this.legendPosition = legendPosition;
	}
	public boolean isStacked() {
		return stacked;
	}
	public void setStacked(boolean stacked) {
		this.stacked = stacked;
	}

}
